<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::with('category')
            ->orderBy('order')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('admin.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::where('is_active', true)
            ->orderBy('order')
            ->get();
        
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'category_id' => 'nullable|exists:categories,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'description_detail' => 'nullable|string',
            'delivery_info' => 'nullable|string',
            'certificates_info' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'sale_price' => 'nullable|numeric|min:0|lt:price',
            'sku' => 'nullable|string|max:255|unique:products,sku',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'gallery.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'stock_quantity' => 'nullable|integer|min:0',
            'in_stock' => 'boolean',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'is_organic' => 'boolean',
            'brand' => 'nullable|string|max:255',
            'farm_name' => 'nullable|string|max:255',
            'origin' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'rating' => 'nullable|integer|min:0|max:5',
            'review_count' => 'nullable|integer|min:0',
            'order' => 'nullable|integer|min:0',
        ]);

        $validated['slug'] = Str::slug($validated['name']);
        
        // Ensure unique slug
        $originalSlug = $validated['slug'];
        $counter = 1;
        while (Product::where('slug', $validated['slug'])->exists()) {
            $validated['slug'] = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Handle main image upload
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = 'product-' . time() . '-' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/products'), $filename);
            $validated['image'] = '/uploads/products/' . $filename;
        }

        // Handle gallery images upload
        if ($request->hasFile('gallery')) {
            $gallery = [];
            foreach ($request->file('gallery') as $file) {
                $filename = 'product-gallery-' . time() . '-' . Str::random(10) . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads/products/gallery'), $filename);
                $gallery[] = '/uploads/products/gallery/' . $filename;
            }
            $validated['gallery'] = $gallery;
        }

        // Set defaults
        $validated['in_stock'] = $request->has('in_stock') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['is_featured'] = $request->has('is_featured') ? true : false;
        $validated['is_organic'] = $request->has('is_organic') ? true : false;
        $validated['rating'] = $validated['rating'] ?? 0;
        $validated['review_count'] = $validated['review_count'] ?? 0;
        $validated['order'] = $validated['order'] ?? 0;

        Product::create($validated);

        return redirect()->route('admin.products.index')
            ->with('success', 'Ürün başarıyla oluşturuldu.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        $product->load('category');
        return view('admin.products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = Category::where('is_active', true)
            ->orderBy('order')
            ->get();
        
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'category_id' => 'nullable|exists:categories,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'description_detail' => 'nullable|string',
            'delivery_info' => 'nullable|string',
            'certificates_info' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'sale_price' => 'nullable|numeric|min:0|lt:price',
            'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'gallery.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'stock_quantity' => 'nullable|integer|min:0',
            'in_stock' => 'boolean',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'is_organic' => 'boolean',
            'brand' => 'nullable|string|max:255',
            'farm_name' => 'nullable|string|max:255',
            'origin' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'rating' => 'nullable|integer|min:0|max:5',
            'review_count' => 'nullable|integer|min:0',
            'order' => 'nullable|integer|min:0',
        ]);

        // Update slug if name changed
        if ($product->name !== $validated['name']) {
            $validated['slug'] = Str::slug($validated['name']);
            // Ensure unique slug
            $originalSlug = $validated['slug'];
            $counter = 1;
            while (Product::where('slug', $validated['slug'])->where('id', '!=', $product->id)->exists()) {
                $validated['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        // Handle main image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($product->image && file_exists(public_path($product->image))) {
                unlink(public_path($product->image));
            }
            
            $file = $request->file('image');
            $filename = 'product-' . time() . '-' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/products'), $filename);
            $validated['image'] = '/uploads/products/' . $filename;
        }

        // Handle gallery images upload
        if ($request->hasFile('gallery')) {
            // Delete old gallery images if exists
            if ($product->gallery) {
                foreach ($product->gallery as $oldImage) {
                    if (file_exists(public_path($oldImage))) {
                        unlink(public_path($oldImage));
                    }
                }
            }
            
            $gallery = [];
            foreach ($request->file('gallery') as $file) {
                $filename = 'product-gallery-' . time() . '-' . Str::random(10) . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads/products/gallery'), $filename);
                $gallery[] = '/uploads/products/gallery/' . $filename;
            }
            $validated['gallery'] = $gallery;
        }

        // Set defaults
        $validated['in_stock'] = $request->has('in_stock') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['is_featured'] = $request->has('is_featured') ? true : false;
        $validated['is_organic'] = $request->has('is_organic') ? true : false;

        $product->update($validated);

        return redirect()->route('admin.products.index')
            ->with('success', 'Ürün başarıyla güncellendi.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete main image if exists
        if ($product->image && file_exists(public_path($product->image))) {
            unlink(public_path($product->image));
        }

        // Delete gallery images if exists
        if ($product->gallery) {
            foreach ($product->gallery as $image) {
                if (file_exists(public_path($image))) {
                    unlink(public_path($image));
                }
            }
        }

        $product->delete();

        return redirect()->route('admin.products.index')
            ->with('success', 'Ürün başarıyla silindi.');
    }
}
