<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Menu;
use App\Models\MenuItem;
use App\Services\MenuService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class MenuController extends Controller
{
    public function __construct(private MenuService $menuService)
    {
    }

    public function index()
    {
        $menus = Menu::with(['items' => function ($q) {
            $q->orderBy('order');
        }])->orderBy('id')->get();

        return view('admin.menus.index', compact('menus'));
    }

    public function storeMenu(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'slug' => ['required', 'string', 'max:100', 'alpha_dash', 'unique:menus,slug'],
            'location' => ['nullable', 'string', 'max:100'],
            'is_active' => ['boolean'],
        ]);

        Menu::create($data);

        return redirect()->back()->with('status', 'Menü oluşturuldu.');
    }

    public function storeItem(Request $request, Menu $menu)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:150'],
            'url' => ['nullable', 'string', 'max:255'],
            'route_name' => ['nullable', 'string', 'max:255'],
            'route_params' => ['nullable', 'string'], // JSON string
            'target' => ['nullable', 'string', 'max:20'],
            'parent_id' => ['nullable', Rule::exists('menu_items', 'id')->where('menu_id', $menu->id)],
            'order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['boolean'],
        ]);

        $data['menu_id'] = $menu->id;
        $data['route_params'] = $data['route_params'] ? json_decode($data['route_params'], true) : null;

        MenuItem::create($data);
        $this->menuService->flush($menu->slug);

        return redirect()->back()->with('status', 'Menü öğesi eklendi.');
    }

    public function updateItem(Request $request, MenuItem $item)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:150'],
            'url' => ['nullable', 'string', 'max:255'],
            'route_name' => ['nullable', 'string', 'max:255'],
            'route_params' => ['nullable', 'string'],
            'target' => ['nullable', 'string', 'max:20'],
            'parent_id' => [
                'nullable',
                Rule::exists('menu_items', 'id')->where('menu_id', $item->menu_id),
                Rule::notIn([$item->id]),
            ],
            'order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['boolean'],
        ]);

        $data['route_params'] = $data['route_params'] ? json_decode($data['route_params'], true) : null;

        $item->update($data);
        $this->menuService->flush($item->menu->slug);

        return redirect()->back()->with('status', 'Menü öğesi güncellendi.');
    }

    public function destroyItem(MenuItem $item)
    {
        $slug = $item->menu->slug;
        $item->delete();
        $this->menuService->flush($slug);

        return redirect()->back()->with('status', 'Menü öğesi silindi.');
    }
}


