<?php

namespace App\Http\Controllers;

use App\Models\Review;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReviewController extends Controller
{
    /**
     * Store a newly created review
     */
    public function store(Request $request, Product $product)
    {
        // Kullanıcı giriş yapmış mı kontrol et
        if (!Auth::check()) {
            return back()->withErrors(['error' => 'Yorum yapmak için giriş yapmalısınız.']);
        }

        // Kullanıcı bu ürünü satın almış mı kontrol et
        if (!$product->hasUserPurchased(Auth::id())) {
            return back()->withErrors(['error' => 'Bu ürüne yorum yapmak için önce satın alma yapmalısınız.']);
        }

        // Kullanıcı daha önce bu ürüne yorum yapmış mı kontrol et
        $existingReview = Review::where('product_id', $product->id)
            ->where('user_id', Auth::id())
            ->first();

        if ($existingReview) {
            return back()->withErrors(['error' => 'Bu ürüne zaten yorum yaptınız.']);
        }

        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:1000',
        ]);

        $review = Review::create([
            'product_id' => $product->id,
            'user_id' => Auth::id(),
            'rating' => $validated['rating'],
            'comment' => $validated['comment'] ?? null,
            'is_approved' => true, // Otomatik onay (isteğe bağlı olarak false yapılabilir)
        ]);

        // Ürünün ortalama rating'ini güncelle
        $this->updateProductRating($product);

        return back()->with('success', 'Yorumunuz başarıyla eklendi!');
    }

    /**
     * Update product rating based on approved reviews
     */
    private function updateProductRating(Product $product)
    {
        $reviews = $product->approvedReviews;
        $reviewCount = $reviews->count();
        
        if ($reviewCount > 0) {
            $avgRating = $reviews->avg('rating');
            $product->update([
                'rating' => $avgRating ? round($avgRating) : 0,
                'review_count' => $reviewCount,
            ]);
        } else {
            $product->update([
                'rating' => 0,
                'review_count' => 0,
            ]);
        }
    }
}
